
using System;
using System.Collections.Generic;
using System.IO;
using Unity.PSN.PS5.Aysnc;
using Unity.PSN.PS5.Internal;
using Unity.PSN.PS5.WebApi;
using UnityEngine;

namespace Unity.PSN.PS5.Sessions
{
    /// <summary>
    /// Members added to a game session
    /// </summary>
    public class GameSessionInitMember
    {
        /// <summary>
        /// User Id for local member
        /// </summary>
        public Int32 UserId { get; set; }


        internal Int32 PushCallbackId { get; set; } = WebApiPushEvent.InvalidCallbackId;

        /// <summary>
        /// Member Account id
        /// </summary>
        public UInt64 AccountId { get; internal set; }

        /// <summary>
        /// Members Platform
        /// </summary>
        public SessionPlatforms Platform { get; set; }

        /// <summary>
        /// Session join status of a member.
        /// </summary>
        public GameSessionJoinState JoinState { get; set; } = GameSessionJoinState.NotSet;

        /// <summary>
        /// Custom binary member data
        /// </summary>
        public byte[] CustomData1 { get; set; }

        internal void Serialise(BinaryWriter writer)
        {
            writer.Write(UserId);
            writer.Write(PushCallbackId);
            writer.Write(AccountId);
            writer.Write((Int32)Platform);
            writer.Write((Int32)JoinState);

            Int32 dataSize = CustomData1 != null ? CustomData1.Length : 0;
            writer.Write(dataSize);
            if (dataSize > 0)
            {
                writer.Write(CustomData1);
            }
        }
    }

    /// <summary>
    /// The parameters used when creating a game session
    /// </summary>
    public class GameSessionCreationParams
    {
        /// <summary>
        /// Maximum number of supported players. Min 1, Max 100
        /// </summary>
        public UInt32 MaxPlayers { get; set; } = 1; // Min 1, Max 100

        /// <summary>
        /// Maximum number of supported spectators. Min 0, Max 50
        /// </summary>
        public UInt32 MaxSpectators { get; set; } = 0; // Min , Max 50

        /// <summary>
        /// Information about the platforms with which users can join a Game Session
        /// </summary>
        public SessionPlatforms SupportedPlatforms { get; set; } = SessionPlatforms.PS5;

        /// <summary>
        /// Flag for temporarily prohibiting joining a Game Session. When true, the Game Session cannot be joined.
        /// </summary>
        public bool JoinDisabled { get; set; } = false;

        /// <summary>
        /// Flag that indicates Game Session and Player Session dependency true indicates an association with a Player Session, and false indicates a standalone Game Session. Defaults true
        /// </summary>
        public bool UsePlayerSession { get; set; } = true;

        /// <summary>
        /// Period of validity for a reservation by a member to join a Game Session
        /// </summary>
        public Int32 ReservationTimeoutSeconds { get; set; } = 300;

        /// <summary>
        /// Additional members to add to the game session
        /// </summary>
        public List<GameSessionInitMember> AdditionalMembers;

        /// <summary>
        /// The set of filters to use for the initial user who creates the session. Default to standard set of WebApi notification types. <see cref="GameSessionFilters"/>
        /// </summary>
        public WebApiFilters SessionFilters { get; set; } = GameSessionFilters.DefaultFilters;

        /// <summary>
        /// Custom data. The maximum size is the size yielded from encoding 1024 bytes in Base64.
        /// </summary>
        public byte[] CustomData1 { get; set; }

        /// <summary>
        /// Custom data. The maximum size is the size yielded from encoding 1024 bytes in Base64.
        /// </summary>
        public byte[] CustomData2 { get; set; }

        /// <summary>
        /// Callbacks triggered when WebApi notifications are recieved and processed by the session
        /// </summary>
        public GameSessionCallbacks Callbacks { get; set; } = new GameSessionCallbacks();

        internal void Serialise(BinaryWriter writer)
        {
            writer.Write(MaxPlayers);
            writer.Write(MaxSpectators);
            writer.Write((UInt32)SupportedPlatforms);
            writer.Write(JoinDisabled);
            writer.Write(UsePlayerSession);
            writer.Write(ReservationTimeoutSeconds);

            Int32 numMembers = AdditionalMembers != null ? AdditionalMembers.Count : 0;
            writer.Write(numMembers);
            for (int i = 0; i < numMembers; i++)
            {
                AdditionalMembers[i].Serialise(writer);
            }

            int customData1Size = CustomData1 != null ? CustomData1.Length : 0;
            writer.Write(customData1Size);
            if (customData1Size > 0)
            {
                writer.Write(CustomData1);
            }

            int customData2Size = CustomData2 != null ? CustomData2.Length : 0;
            writer.Write(customData2Size);
            if (customData2Size > 0)
            {
                writer.Write(CustomData2);
            }
        }
    }

    /// <summary>
    /// Requests for Game sessions
    /// </summary>
    public class GameSessionRequests
    {
        internal enum NativeMethods : UInt32
        {
            CreateGameSession = 0x0B00001u,
            LeaveGameSession = 0x0B00002u,
            JoinGameSession = 0x0B00003u,
            GetGameSessions = 0x0B00004u,
            SetGameSessionProperties = 0x0B00005u,
            SetGameSessionMemberSystemProperties = 0x0B00006u,
            SendGameSessionMessage = 0x0B00007u,
            GetJoinedGameSessionsByUser = 0x0B00008u,
            DeleteGameSession = 0x0B00009u,
        }

        /// <summary>
        /// Create a new game session
        /// </summary>
        public class CreateGameSessionRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Custom binary member data for game session creator
            /// </summary>
            public byte[] CreatorsCustomData1 { get; set; }

            /// <summary>
            /// The parameters used to create the session
            /// </summary>
            public GameSessionCreationParams CreationParams { get; set; }

            /// <summary>
            /// The created game session instance
            /// </summary>
            public GameSession Session { get; internal set; }

            protected internal override void Run()
            {
                WebApiPushEvent sessionPushEvent = null;

                Result = WebApiNotifications.CreatePushEventBlocking(UserId, CreationParams.SessionFilters, GameSession.SessionWebApiEventHandler, true, out sessionPushEvent);

                if (sessionPushEvent == null)
                {
                    return;
                }

                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)GameSessionRequests.NativeMethods.CreateGameSession);

                GameSessionInitMember startMember = new GameSessionInitMember();
                startMember.UserId = UserId;
                startMember.PushCallbackId = sessionPushEvent.PushCallbackId;
                startMember.CustomData1 = CreatorsCustomData1;

                startMember.Serialise(nativeMethod.Writer);

                CreationParams.Serialise(nativeMethod.Writer);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    // Read back the results from the native method
                    string sessionId = nativeMethod.Reader.ReadPrxString();

                    bool isNewSession = false;
                    Session = SessionsManager.CreateGameSession(sessionId, out isNewSession);

                    Session.InitialiseCreationParams(CreationParams);
                    Session.Initialise(CreationParams.Callbacks, sessionPushEvent);

                    // Deserialise the players
                    int numPlayers = nativeMethod.Reader.ReadInt32();

                    for (int i = 0; i < numPlayers; i++)
                    {
                        SessionMember newMember = new SessionMember();
                        newMember.DeserialiseBasicInfo(nativeMethod.Reader, false);

                        Session.AddorUpdateMember(newMember);
                    }

                    UInt64 creatorAccountId = nativeMethod.Reader.ReadUInt64();

                    Session.UpdateUserId(creatorAccountId, UserId);
                }
                else
                {
                    // If session failed unregister the user push event
                    WebApiNotifications.UnregisterPushEventCall(sessionPushEvent);
                }
                
                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Get a local user to leave the game session
        /// </summary>
        public class LeaveGameSessionRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Session to leave
            /// </summary>
            public string SessionId { get; set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)GameSessionRequests.NativeMethods.LeaveGameSession);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);
                nativeMethod.Writer.WritePrxString(SessionId);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    UInt64 accountId = nativeMethod.Reader.ReadUInt64();

                    GameSession session = SessionsManager.FindGameSessionFromSessionId(SessionId);

                    if (session != null)
                    {
                        session.RemoveMember(accountId);

                        session.DeletePushEvent(UserId);
                    }
                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Get a user to join the game session
        /// </summary>
        public class JoinGameSessionRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Session ID to join
            /// </summary>
            public string SessionId { get; set; }

            /// <summary>
            /// Join the player session as a spectator
            /// </summary>
            public bool JoinAsSpectator { get; set; } = false;

            /// <summary>
            /// The joined session
            /// </summary>
            public GameSession Session { get; internal set; }

            /// <summary>
            /// The session filters to use when creating the users WebApi push events
            /// </summary>
            public WebApiFilters SessionFilters { get; set; } = GameSessionFilters.DefaultFilters;

            /// <summary>
            /// The callback used when session is updated
            /// </summary>
            public GameSessionCallbacks Callbacks { get; set; } = new GameSessionCallbacks();

            protected internal override void Run()
            {
                WebApiPushEvent sessionPushEvent = null;

                Result = WebApiNotifications.CreatePushEventBlocking(UserId, SessionFilters, GameSession.SessionWebApiEventHandler, true, out sessionPushEvent);

                if (sessionPushEvent == null)
                {
                    return;
                }

                // Find a local session. If one already exists then it means the player is join a local session
                // Must also handle if the session id doesn't exist because the the user might be joining a player session just using the session id.
                bool isNewSession = false;
                Session = SessionsManager.CreateGameSession(SessionId, out isNewSession);

                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)GameSessionRequests.NativeMethods.JoinGameSession);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);
                nativeMethod.Writer.Write(sessionPushEvent.PushCallbackId);
                nativeMethod.Writer.Write(JoinAsSpectator);
                nativeMethod.Writer.Write(false);
                nativeMethod.Writer.WritePrxString(SessionId);

                // Debug.LogError("nativeMethod Call");
                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    Session.Initialise(Callbacks, sessionPushEvent);

                    // Deserialise the players
                    int numPlayers = nativeMethod.Reader.ReadInt32();

                    //    Debug.LogError("numPlayers " + numPlayers);
                    for (int i = 0; i < numPlayers; i++)
                    {
                        SessionMember newMember = new SessionMember();
                        newMember.DeserialiseBasicInfo(nativeMethod.Reader, JoinAsSpectator);

                        Session.AddorUpdateMember(newMember);
                    }

                    UInt64 joinerAccountId = nativeMethod.Reader.ReadUInt64();

                    Session.UpdateUserId(joinerAccountId, UserId);
                }
                else
                {
                    // If session failed unregister the user push event
                    WebApiNotifications.UnregisterPushEventCall(sessionPushEvent);
                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Retrieved session data when using <see cref="GetGameSessionsRequest"/>
        /// </summary>
        public class RetrievedSessionData
        {
            /// <summary>
            /// The flags represent which parts of the data have been returned based on the <see cref="GetGameSessionsRequest.RequiredFields"/>
            /// </summary>
            public GameSession.ParamTypes SetFlags { get; internal set; }

            /// <summary>
            /// True is the calling user is in the session
            /// </summary>
            public bool IsUserInSession { get; internal set; }

            /// <summary>
            /// The session id
            /// </summary>
            public string SessionId { get; internal set; }

            /// <summary>
            /// Date and time of creation of the Player Session
            /// </summary>
            public DateTime CreatedTimeStamp { get; internal set; }

            /// <summary>
            /// Maximum number of members who can join as players
            /// </summary>
            public UInt32 MaxPlayers { get; internal set; }

            /// <summary>
            /// Maximum number of members who can join as spectators
            /// </summary>
            public UInt32 MaxSpectators { get; internal set; }

            /// <summary>
            /// Members participating as players. This will only contain a partial set of session member data
            /// </summary>
            public SessionMember[] Players { get; internal set; }

            /// <summary>
            /// Members participating as spectators. This will only contain a partial set of session member data
            /// </summary>
            public SessionMember[] Spectators { get; internal set; }

            /// <summary>
            /// Flag for temporarily prohibiting joining
            /// </summary>
            public bool JoinDisabled { get; internal set; }

            /// <summary>
            /// Platforms that can join
            /// </summary>
            public SessionPlatforms SupportedPlatforms { get; internal set; }

            /// <summary>
            /// Game Session representative
            /// </summary>
            public GameSessionRepresentative Representative { get; internal set; }

            /// <summary>
            /// Custom data. The maximum size is the size yielded from encoding 1024 bytes in Base64.
            /// </summary>
            public byte[] CustomData1 { get; internal set; }

            /// <summary>
            /// Custom data. The maximum size is the size yielded from encoding 1024 bytes in Base64.
            /// </summary>
            public byte[] CustomData2 { get; internal set; }

            /// <summary>
            /// Flag that indicates Game Session and Player Session dependency.
            /// true indicates an association with a Player Session, and false indicates a standalone Game Session.
            /// </summary>
            public bool UsePlayerSession { get; internal set; }

            /// <summary>
            /// Information added by the Matchmaking service
            /// </summary>
            public string MatchmakingOfferId { get; internal set; }

            /// <summary>
            /// Period of validity for a reservation by a member to join a Game Session
            /// </summary>
            public Int32 ReservationTimeoutSeconds { get; internal set; }


            internal void Deserialise(BinaryReader reader)
            {
                SetFlags = 0;

                IsUserInSession = reader.ReadBoolean();

                if (IsUserInSession == false) return;

                int numSessions = reader.ReadInt32();

                bool isValid = reader.ReadBoolean();

                if (isValid == false) return;

                bool isSessionSet = reader.ReadBoolean();
                if (isSessionSet)
                {
                    SessionId = reader.ReadPrxString();
                    SetFlags |= GameSession.ParamTypes.SessionId;
                }

                bool isCreatedTimeStamp = reader.ReadBoolean();
                if (isCreatedTimeStamp)
                {
                    CreatedTimeStamp = reader.ReadUnixTimestampString();
                    SetFlags |= GameSession.ParamTypes.CreatedTimeStamp;
                }

                bool isMaxPlayers = reader.ReadBoolean();
                if (isMaxPlayers)
                {
                    MaxPlayers = reader.ReadUInt32();
                    SetFlags |= GameSession.ParamTypes.MaxPlayers;
                }

                bool isMaxSpectators = reader.ReadBoolean();
                if (isMaxSpectators)
                {
                    MaxSpectators = reader.ReadUInt32();
                    SetFlags |= GameSession.ParamTypes.MaxSpectators;
                }

                bool memberSet = reader.ReadBoolean();

                if (memberSet == true)
                {
                    SetFlags |= GameSession.ParamTypes.Member;

                    bool playersSet = reader.ReadBoolean();

                    if (playersSet == true)
                    {
                        SetFlags |= GameSession.ParamTypes.MemberPlayers;

                        int numPlayers = reader.ReadInt32();

                        Players = new SessionMember[numPlayers];

                        for (int i = 0; i < numPlayers; i++)
                        {
                            Players[i] = new SessionMember();
                            Players[i].Deserialise(reader);

                            bool isMemberPlayersJoinState = reader.ReadBoolean();
                            if (isMemberPlayersJoinState)
                            {
                                SetFlags |= GameSession.ParamTypes.MemberPlayersJoinState;
                                Players[i].DeserialiseJoinState(reader);
                            }

                            bool isCustomData1Set = reader.ReadBoolean();
                            if (isCustomData1Set)
                            {
                                SetFlags |= GameSession.ParamTypes.MemberPlayersCustomData1;
                                Players[i].DeserialiseCustomData(reader);
                            }
                        }
                    }

                    bool spectatorsSet = reader.ReadBoolean();

                    if (spectatorsSet == true)
                    {
                        SetFlags |= GameSession.ParamTypes.MemberSpectators;

                        int numSpectators = reader.ReadInt32();

                        Spectators = new SessionMember[numSpectators];

                        for (int i = 0; i < numSpectators; i++)
                        {
                            Spectators[i] = new SessionMember();
                            Spectators[i].Deserialise(reader);

                            bool isMemberSpectatorsJoinState = reader.ReadBoolean();
                            if (isMemberSpectatorsJoinState)
                            {
                                SetFlags |= GameSession.ParamTypes.MemberSpectatorsJoinState;
                                Spectators[i].DeserialiseJoinState(reader);
                            }

                            bool isCustomData1Set = reader.ReadBoolean();
                            if (isCustomData1Set)
                            {
                                SetFlags |= GameSession.ParamTypes.MemberSpectatorsCustomData1;
                                Spectators[i].DeserialiseCustomData(reader);
                            }
                        }
                    }
                }

                bool isJoinDisabled = reader.ReadBoolean();
                if (isJoinDisabled)
                {
                    SetFlags |= GameSession.ParamTypes.JoinDisabled;
                    JoinDisabled = reader.ReadBoolean();
                }

                bool isSupportedPlatforms = reader.ReadBoolean();
                if (isSupportedPlatforms)
                {
                    SetFlags |= GameSession.ParamTypes.SupportedPlatforms;
                    SupportedPlatforms = (SessionPlatforms)reader.ReadUInt32();
                }

                bool isRepresentativeSet = reader.ReadBoolean();
                if (isRepresentativeSet)
                {
                    SetFlags |= GameSession.ParamTypes.Representative;
                    if (Representative == null)
                    {
                        Representative = new GameSessionRepresentative();
                    }

                    Representative.Deserialise(reader);
                }

                bool isCustomData1 = reader.ReadBoolean();
                if (isCustomData1)
                {
                    CustomData1 = reader.ReadData();
                    SetFlags |= GameSession.ParamTypes.CustomData1;
                }

                bool isCustomData2 = reader.ReadBoolean();
                if (isCustomData2)
                {
                    CustomData2 = reader.ReadData();
                    SetFlags |= GameSession.ParamTypes.CustomData2;
                }

                bool isUsePlayerSession = reader.ReadBoolean();
                if (isUsePlayerSession)
                {
                    UsePlayerSession = reader.ReadBoolean();
                    SetFlags |= GameSession.ParamTypes.UsePlayerSession;
                }

                bool isMatchmakingOfferId = reader.ReadBoolean();
                if (isMatchmakingOfferId)
                {
                    MatchmakingOfferId = reader.ReadPrxString();
                    SetFlags |= GameSession.ParamTypes.Matchmaking;
                }

                bool isReservationTimeoutSeconds = reader.ReadBoolean();
                if (isReservationTimeoutSeconds)
                {
                    ReservationTimeoutSeconds = reader.ReadInt32();
                    SetFlags |= GameSession.ParamTypes.ReservationTimeoutSeconds;
                }
            }
        }

        /// <summary>
        /// Get info for a session
        /// </summary>
        public class GetGameSessionsRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Session ID to retrieve info
            /// </summary>
            public string SessionIds { get; set; }

            /// <summary>
            /// Which fields should be retrieved for the session
            /// </summary>
            public GameSession.ParamTypes RequiredFields { get; set; } = GameSession.ParamTypes.Default;

            /// <summary>
            /// The retieved session data
            /// </summary>
            public RetrievedSessionData SessionData { get; internal set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.GetGameSessions);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);
                nativeMethod.Writer.WritePrxString(SessionIds);

                // Must add SessionId flag so when the data is returned the session id is included
                GameSession.ParamTypes required = RequiredFields | GameSession.ParamTypes.SessionId;

                SessionFieldFlagsSerialiser.SerialiseFieldFlags((UInt32)required, GameSession.FlagText, nativeMethod.Writer);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    SessionData = new RetrievedSessionData();

                    SessionData.Deserialise(nativeMethod.Reader);
                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Update Game Session information
        /// </summary>
        /// <remarks>
        /// Modify one item of information specified, out of all the information concerning the Game Session (it is not possible to update multiple items, and specifying multiple such items will result in an error).
        /// If <see cref="MaxPlayers"/> is being updated, it cannot be modified to a value smaller than the number of players already currently participating.
        /// If <see cref="MaxSpectators"/> is being updated, it cannot be modified to a value smaller than the number of spectators already currently participating.
        /// </remarks>
        public class SetGameSessionPropertiesRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Session id to update
            /// </summary>
            public string SessionId { get; set; }

            /// <summary>
            /// Which parameters should be set. Only one item can be set at a time, specifying multiple such items will result in an error
            /// </summary>
            public GameSession.ParamTypes ParamToSet { get; set; }

            /// <summary>
            /// Maximum number of supported players. Min 1, Max 100
            /// </summary>
            public UInt32 MaxPlayers { get; set; } = 1; // Min 1, Max 100

            /// <summary>
            /// Maximum number of supported spectators. Min 0, Max 50
            /// </summary>
            public UInt32 MaxSpectators { get; set; } = 0; // Min , Max 50

            /// <summary>
            /// Flag that temporarily prohibits joining a Game Session.
            /// </summary>
            public bool JoinDisabled { get; set; } = false;

            /// <summary>
            /// Custom binary session data
            /// </summary>
            public byte[] CustomData1 { get; set; }

            /// <summary>
            /// Custom binary session data
            /// </summary>
            public byte[] CustomData2 { get; set; }



            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.SetGameSessionProperties);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);

                nativeMethod.Writer.WritePrxString(SessionId);

                if ((ParamToSet & GameSession.ParamTypes.MaxPlayers) != 0)
                {
                    nativeMethod.Writer.Write(true);
                    nativeMethod.Writer.Write(MaxPlayers);
                }
                else
                {
                    nativeMethod.Writer.Write(false);
                }

                if ((ParamToSet & GameSession.ParamTypes.MaxSpectators) != 0)
                {
                    nativeMethod.Writer.Write(true);
                    nativeMethod.Writer.Write(MaxSpectators);
                }
                else
                {
                    nativeMethod.Writer.Write(false);
                }

                if ((ParamToSet & GameSession.ParamTypes.JoinDisabled) != 0)
                {
                    nativeMethod.Writer.Write(true);
                    nativeMethod.Writer.Write(JoinDisabled);
                }
                else
                {
                    nativeMethod.Writer.Write(false);
                }

                if ((ParamToSet & GameSession.ParamTypes.CustomData1) != 0)
                {
                    nativeMethod.Writer.Write(true);
                    nativeMethod.Writer.Write(CustomData1.Length);
                    nativeMethod.Writer.Write(CustomData1);
                }
                else
                {
                    nativeMethod.Writer.Write(false);
                }

                if ((ParamToSet & GameSession.ParamTypes.CustomData2) != 0)
                {
                    nativeMethod.Writer.Write(true);
                    nativeMethod.Writer.Write(CustomData2.Length);
                    nativeMethod.Writer.Write(CustomData2);
                }
                else
                {
                    nativeMethod.Writer.Write(false);
                }

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {

                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Set member system properties for a game session
        /// </summary>
        public class SetGameSessionMemberSystemPropertiesRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Game Session ID
            /// </summary>
            public string SessionId { get; set; }

            /// <summary>
            /// Custom data. The maximum size is the size yielded from encoding 1024 bytes in Base64
            /// </summary>
            public byte[] CustomData1 { get; set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.SetGameSessionMemberSystemProperties);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);

                nativeMethod.Writer.WritePrxString(SessionId);

                nativeMethod.Writer.Write(CustomData1.Length);
                nativeMethod.Writer.Write(CustomData1);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {

                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Send gamegame session message
        /// </summary>
        public class SendGameSessionMessageRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// List of recipients' Account IDs
            /// </summary>
            public List<UInt64> ToAccountId { get; set; }

            /// <summary>
            /// Game Session ID
            /// </summary>
            public string SessionId { get; set; }

            /// <summary>
            /// Specify an arbitrary string (e.g., in Base64 or JSON). The amount of data must be at least 1 byte and no greater than 8032 bytes
            /// </summary>
            public string Payload { get; set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.SendGameSessionMessage);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);

                nativeMethod.Writer.WritePrxString(SessionId);

                nativeMethod.Writer.WritePrxString(Payload);

                int count = ToAccountId != null ? ToAccountId.Count : 0;

                nativeMethod.Writer.Write(count);

                for (int i = 0; i < count; i++)
                {
                    nativeMethod.Writer.Write(ToAccountId[i]);
                }

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {

                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Joined game session info
        /// </summary>
        public class JoinedGameSession
        {
            /// <summary>
            /// The session id joined by the user
            /// </summary>
            public string SessionId { get; internal set; }

            /// <summary>
            /// The platform for the session
            /// </summary>
            public SessionPlatforms Platform { get; internal set; }

            internal void Deserialise(BinaryReader reader)
            {
                SessionId = reader.ReadPrxString();
                Platform = (SessionPlatforms)reader.ReadUInt32();
            }
        }

        /// <summary>
        /// Obtain a list of Player Sessions in which a user is participating
        /// </summary>
        public class GetJoinedGameSessionsByUserRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// List of Player Sessions obtained
            /// </summary>
            public List<JoinedGameSession> FoundPlayerSessions { get; internal set; } = new List<JoinedGameSession>();

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.GetJoinedGameSessionsByUser);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);

                FoundPlayerSessions.Clear();

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    Int32 count = nativeMethod.Reader.ReadInt32();

                    for (int i = 0; i < count; i++)
                    {
                        JoinedGameSession jps = new JoinedGameSession();
                        jps.Deserialise(nativeMethod.Reader);
                        FoundPlayerSessions.Add(jps);
                    }
                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

        /// <summary>
        /// Delete game session
        /// </summary>
        public class DeleteGameSessionRequest : Request
        {
            /// <summary>
            /// User ID
            /// </summary>
            public Int32 UserId { get; set; }

            /// <summary>
            /// Game Session ID
            /// </summary>
            public string SessionId { get; set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.DeleteGameSession);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(UserId);

                nativeMethod.Writer.WritePrxString(SessionId);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {

                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }
    }

}