
using System;
using System.IO;
using Unity.PSN.PS5.Aysnc;
using Unity.PSN.PS5.Internal;
using System.Collections.Generic;

#if UNITY_PS5
namespace Unity.PSN.PS5.Entitlement
{
    /// <summary>
    /// Entitlements provides features for accessing additional content, consumable entitlements, and subscription entitlements. 
    /// </summary>
    public class Entitlements
    {
        enum NativeMethods : UInt32
        {
            GetAdditionalContentEntitlementList = 0xFA00001u,
        }

        static WorkerThread workerQueue = new WorkerThread();

        internal static void Start()
        {
            workerQueue.Start("Entitlements");
        }

        internal static void Stop()
        {
            workerQueue.Stop();
        }

        /// <summary>
        /// Schedule an <see cref="AsyncOp"/> by adding it to the internal Entitlements queue
        /// </summary>
        /// <param name="op">The operation to schedule</param>
        public static void Schedule(AsyncOp op)
        {
            workerQueue.Schedule(op);
        }

        /// <summary>
        /// Package type
        /// </summary>
        public enum EntitlementAccessPackageType
        {
            /// <summary> Undefined type </summary>
            None = 0,
            /// <summary> Application </summary>
            PSGD,
            /// <summary> Additional content with extra data </summary>
            PSAC,
            /// <summary> Additional content without extra data </summary>
            PSAL
        }

        /// <summary>
        /// Download status
        /// </summary>
        public enum EntitlementAccessDownloadStatus
        {
            /// <summary> This additional content does not include any data to be downloaded </summary>
            NoExtraData = 0,
            /// <summary> There is data to be downloaded, but it has not been installed </summary>
            NoInQueue,
            /// <summary> There is data to be downloaded, and it is currently being downloaded </summary>
            Downloading,
            /// <summary> There is data to be downloaded, and the download is currently paused </summary>
            DownloadSuspended,
            /// <summary> There is data to be downloaded, and installation is complete </summary>
            Installed,
        }

        /// <summary>
        /// Additional content information
        /// </summary>
        public class AdditionalContentEntitlementInfo
        {
            /// <summary>
            /// Unified entitlement label
            /// </summary>
            public string EntitlementLabel { get; internal set; }

            /// <summary>
            /// Package type
            /// </summary>
            public EntitlementAccessPackageType PackageType { get; internal set; }

            /// <summary>
            /// Download status
            /// </summary>
            public EntitlementAccessDownloadStatus DownloadStatus { get; internal set; }

            internal void Deserialise(BinaryReader reader)
            {
                EntitlementLabel = reader.ReadPrxString();
                PackageType = (EntitlementAccessPackageType)reader.ReadUInt32();
                DownloadStatus = (EntitlementAccessDownloadStatus)reader.ReadUInt32();
            }
        }

        /// <summary>
        /// Gets a list of additional content information for which the entitlement is valid
        /// </summary>
        public class GetAdditionalContentEntitlementListRequest : Request
        {
            /// <summary>
            /// Service Label used when register the filters
            /// </summary>
            public UInt32 ServiceLabel { get; set; } = UInt32.MaxValue;

            /// <summary>
            /// List of additional content entitlements.
            /// </summary>
            public AdditionalContentEntitlementInfo[] Entitlements { get; internal set; }

            protected internal override void Run()
            {
                MarshalMethods.MethodHandle nativeMethod = MarshalMethods.PrepareMethod((UInt32)NativeMethods.GetAdditionalContentEntitlementList);

                // Write the data to match the expected format in the native code
                nativeMethod.Writer.Write(ServiceLabel);

                nativeMethod.Call();

                Result = nativeMethod.callResult;

                if (Result.apiResult == APIResultTypes.Success)
                {
                    // Read back the results from the native method
                    int numEntitlements = nativeMethod.Reader.ReadInt32();

                    if (numEntitlements == 0)
                    {
                        Entitlements = null;
                    }
                    else
                    {
                        Entitlements = new AdditionalContentEntitlementInfo[numEntitlements];

                        for(int i = 0; i < numEntitlements; i++)
                        {
                            Entitlements[i] = new AdditionalContentEntitlementInfo();

                            Entitlements[i].Deserialise(nativeMethod.Reader);
                        }
                    }
                }

                MarshalMethods.ReleaseHandle(nativeMethod);
            }
        }

    }
}
#endif